//Water vertex shader - calculates output positions, the camera's forward vector, the sun's light vector and the reflection position to be used for reflection
cbuffer MatrixBuffer : register(b0)
{
	matrix worldMatrix;
	matrix viewMatrix;
	matrix projectionMatrix;
};

cbuffer WaterVertexBuffer : register(b1)
{
	float4 cameraPosition;
	matrix reflectionMatrix;	//The matrix for reflection
	float3 sunPosition;
	float padding;
};

struct InputType
{
	float4 position : POSITION;
	float2 tex : TEXCOORD0;
};

struct OutputType
{
	float4 position : SV_POSITION;
	float2 tex : TEXCOORD0;
	float3 cameraForwardVector : TEXCOORD1;
	float3 lightVector : TEXCOORD2;	//Vector from the sun
	float4 reflectionPosition : TEXCOORD3;
};

OutputType main(InputType input)
{
	OutputType output;

	input.position.w = 1.0f;

	//Calculate the position of the vertex against the world, view, and projection matrices.
	output.position = mul(input.position, worldMatrix);
	output.position = mul(output.position, viewMatrix);
	output.position = mul(output.position, projectionMatrix);

	//Store the texture coordinates for the pixel shader.
	output.tex = input.tex;

	//Calculate the camera's forward vector
	output.cameraForwardVector = cameraPosition.xyz - mul(input.position, worldMatrix);

	//Calculate the sun's vector
	output.lightVector = mul(input.position, worldMatrix) - sunPosition;

	//Calculate the reflection projection world matrix to calculate the reflection position
	matrix reflectProjectWorld = mul(reflectionMatrix, projectionMatrix);
	reflectProjectWorld = mul(worldMatrix, reflectProjectWorld);

	//Calculate the reflection position against the reflectProjectWorld matrix
	output.reflectionPosition = mul(input.position, reflectProjectWorld);

	return output;
}